/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY.                         *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.  Contact information: bergmark@cs.cornell.edu     *
 ******************************************************************************/

package cnrg.itx.gtwy;

import cnrg.itx.datax.*;

/**
 * Takes care of servicing incoming calls from the phone.  Sets up a connection with a computer if a call
 * has been accepted.  Services the next incoming phone call when the connection ends or the previous
 * call has been rejected.
 * 
 * @author James Wann
 * @version 1.0a
 */
public class ListenChannel extends Thread
{
	static
	{
		System.loadLibrary("Dialogic");
	}
	
	/**
	 * The Line instance corresponding to ListenChannel's line.
	 */
	private Line myLine;
	
	/**
	 * The Gateway instance.
	 */
	private Gateway myGate;
	
	/**
	 * @param lin the <code>Line</code> instance corresponding to <code>ListenChannel's</code> line.
	 * Provides information concerning that line.
	 * @param g the <code>Gateway</code> instance to access line state.
	 * @see Line
	 * @see Gateway
	 */
	protected ListenChannel(Line lin, Gateway g)
	{
		myLine = lin;
		myGate = g;
	}
	
	/**
	 * Puts the <code>ListenChannel</code> to "sleep".  Used when a connection is being set up.
	 * @see Gateway#setUpConnection(Line, Connection)
	 */
	protected synchronized void makeWait()
	{
		try
		{
			wait();
		}
		catch (InterruptedException e)
		{
		}
	}
	
	/**
	 * "Wakes up" the <code>ListenChannel</code> to listen for the next incoming call.
	 */
	protected synchronized void restart()
	{
		notify();
	}
	
	/**
	 * @see java.lang.Runnable
	 */
	public void run()
	{
		int chDev = myLine.getChDev();
		int index = myLine.getLineNumber() - 1;
		String extension = new String();					// an extension number
		Connection isApproved = null;						// data connection if call is approved
		String errorString = new String("<GatewayException> :: Error: No record found");
		
		while (true)
		{
			System.out.println("<Channel: " + myLine.getLineNumber() + "> waiting for a call.");
			
			waitForCall(chDev);
			
			// Try to reserve ListenChannel's corresponding line.  If so, get the extension number.
			if (myGate.useLine(myLine))
			{
				System.out.println("<Channel: " + myLine.getLineNumber() + "> leave your extension number.");
				extension = getExtension(chDev, false);
				isApproved = null;
				
				// Dial up extension number
				while (isApproved == null)
				{
					try
					{
						isApproved = myGate.dialComputer(myLine, extension);
					}
					catch (GatewayException e)
					{
						System.out.println("<Channel: " + myLine.getLineNumber() + "> -> " + e.getMessage());
						
						// If number is not in the directory then get another extension number
						if (e.getMessage().equals(errorString))
							extension = getExtension(chDev, true);
						else
						{
							// If call could not go through, hang up and listen for another call
							if (extension != null)
								rejectCallMessage(chDev);
							myGate.freeLine(myLine);
							break;
						}
					}
				}
				
				// set up connection if extension number is valid
				if (isApproved != null)
					myGate.setUpConnection(myLine, isApproved);
			}
		}
	}
	
	/**
	 * Sets up the channel to wait for a phone to dial in.
	 * 
	 * @param chDev the channel's device handler.
	 */
	private native void waitForCall(int chDev);
	
	/**
	 * This is where the channel gets the phone's requested extension number.  Number corresponds with
	 * whomever phone user wishes to contact.
	 * 
	 * @param chDev the channel's device handler.
	 * @param isSecondTime true if this is not the first call to getExtension during session.
	 * @return the extension number.
	 */
	private native String getExtension(int chDev, boolean isSecondTime);
	
	/**
	 * Plays out a message to a phone when a call has been rejected.
	 * 
	 * @param chDev the channel's device handler.
	 */
	private native void rejectCallMessage(int chDev);
}